<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/razorpay_config.php';
require_once __DIR__ . '/classes/Order.php';
require_once __DIR__ . '/classes/Cart.php';

// Start session
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check if order ID is provided
if (!isset($_GET['order_id'])) {
    header('Location: cart.php');
    exit;
}

$orderId = (int)$_GET['order_id'];
$order = Order::getOrderWithItems($orderId);

if (!$order) {
    header('Location: cart.php');
    exit;
}

// Verify order belongs to user (if logged in)
if (isset($_SESSION['user_id'])) {
    if (!Order::validateOrderOwnership($orderId, $_SESSION['user_id'])) {
        header('Location: cart.php');
        exit;
    }
}

// Get Razorpay configuration
$razorpay_keys = getRazorpayKeys();
$razorpay_key_id = $razorpay_keys['key_id'];
$razorpay_key_secret = $razorpay_keys['key_secret'];

// Create Razorpay order
function createRazorpayOrder($amount, $currency = 'INR') {
    global $razorpay_key_id, $razorpay_key_secret;
    
    // Check if keys are configured
    if (strpos($razorpay_key_id, 'YOUR_') !== false || strpos($razorpay_key_secret, 'YOUR_') !== false) {
        error_log('Razorpay API keys not configured properly');
        return ['error' => 'API keys not configured. Please contact administrator.'];
    }
    
    $url = 'https://api.razorpay.com/v1/orders';
    $data = [
        'amount' => $amount * 100, // Razorpay expects amount in paise
        'currency' => $currency,
        'receipt' => 'order_' . time()
    ];
    
    $headers = [
        'Content-Type: application/json',
        'Authorization: Basic ' . base64_encode($razorpay_key_id . ':' . $razorpay_key_secret)
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Log the response for debugging
    error_log('Razorpay API Response: ' . $response);
    error_log('Razorpay API HTTP Code: ' . $httpCode);
    
    if ($curlError) {
        error_log('Curl Error: ' . $curlError);
        return ['error' => 'Network error: ' . $curlError];
    }
    
    if ($httpCode === 200) {
        $result = json_decode($response, true);
        if ($result && isset($result['id'])) {
            return $result;
        } else {
            error_log('Invalid response format: ' . $response);
            return ['error' => 'Invalid response from payment gateway'];
        }
    } else {
        error_log('Razorpay API Error: HTTP ' . $httpCode . ' - ' . $response);
        return ['error' => 'Payment gateway error (HTTP ' . $httpCode . ')'];
    }
}

// Handle payment verification
if (isset($_POST['razorpay_payment_id'])) {
    $razorpay_payment_id = $_POST['razorpay_payment_id'];
    $razorpay_order_id = $_POST['razorpay_order_id'];
    $razorpay_signature = $_POST['razorpay_signature'];
    
    // In test mode, accept all payments
    if (RAZORPAY_TEST_MODE) {
        $payment_valid = true;
        error_log('Test mode: Accepting payment without signature verification');
    } else {
        // Verify payment signature
        $expected_signature = hash_hmac('sha256', $razorpay_order_id . '|' . $razorpay_payment_id, $razorpay_key_secret);
        $payment_valid = ($expected_signature === $razorpay_signature);
    }
    
    if ($payment_valid) {
        // Payment successful
        Order::updatePaymentStatus($orderId, 'paid', $razorpay_order_id, $razorpay_payment_id);
        Order::updateOrderStatus($orderId, 'processing');
        
        // Clear cart
        $cart = new Cart();
        $cart->clear();
        
        // Redirect to success page
        header('Location: payment_success.php?order_id=' . $orderId);
        exit;
    } else {
        // Payment failed
        Order::updatePaymentStatus($orderId, 'failed');
        header('Location: payment_failed.php?order_id=' . $orderId);
        exit;
    }
}

// Create Razorpay order
if (RAZORPAY_TEST_MODE) {
    // Use mock order for testing
    $razorpayOrder = createMockRazorpayOrder($order['total_amount']);
    error_log('Using mock Razorpay order: ' . $razorpayOrder['id']);
} else {
    // Use real Razorpay API
    $razorpayOrder = createRazorpayOrder($order['total_amount']);
}

if (isset($razorpayOrder['error'])) {
    // Log the error
    error_log('Payment order creation failed: ' . $razorpayOrder['error']);
    
    // Show user-friendly error page
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Payment Error - PoojaFestival.in</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
        <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.7.2/font/bootstrap-icons.css" rel="stylesheet">
    </head>
    <body>
        <div class="container my-5">
            <div class="row justify-content-center">
                <div class="col-md-6">
                    <div class="card border-danger">
                        <div class="card-body text-center">
                            <i class="bi bi-exclamation-triangle text-danger" style="font-size: 4rem;"></i>
                            <h3 class="mt-3">Payment Setup Error</h3>
                            <p class="text-muted"><?php echo htmlspecialchars($razorpayOrder['error']); ?></p>
                            <div class="mt-4">
                                <a href="checkout.php" class="btn btn-primary">Back to Checkout</a>
                                <a href="cart.php" class="btn btn-outline-secondary">Back to Cart</a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Update order with Razorpay order ID
Order::updatePaymentStatus($orderId, 'pending', $razorpayOrder['id']);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment - PoojaFestival.in</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.7.2/font/bootstrap-icons.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="https://checkout.razorpay.com/v1/checkout.js"></script>
</head>
<body>
    <!-- Navigation -->
    <?php include 'includes/nav.php'; ?>

    <div class="container my-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header">
                        <h4 class="mb-0"><i class="bi bi-credit-card"></i> Complete Payment</h4>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h5>Order Summary</h5>
                                <p><strong>Order ID:</strong> #<?php echo $orderId; ?></p>
                                <p><strong>Total Amount:</strong> ₹<?php echo number_format($order['total_amount'], 2); ?></p>
                                
                                <div class="mt-3">
                                    <h6>Items:</h6>
                                    <?php foreach ($order['items'] as $item): ?>
                                    <div class="d-flex justify-content-between mb-1">
                                        <span><?php echo htmlspecialchars($item['name']); ?> (Qty: <?php echo $item['quantity']; ?>)</span>
                                        <span>₹<?php echo number_format($item['price'] * $item['quantity'], 2); ?></span>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <h5>Payment Options</h5>
                                <p class="text-muted">Choose your preferred payment method:</p>
                                
                                <div class="payment-methods">
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="radio" name="payment_method" id="card" value="card" checked>
                                        <label class="form-check-label" for="card">
                                            <i class="bi bi-credit-card text-primary"></i> Credit/Debit Card
                                        </label>
                                    </div>
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="radio" name="payment_method" id="upi" value="upi">
                                        <label class="form-check-label" for="upi">
                                            <i class="bi bi-phone text-success"></i> UPI
                                        </label>
                                    </div>
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="radio" name="payment_method" id="netbanking" value="netbanking">
                                        <label class="form-check-label" for="netbanking">
                                            <i class="bi bi-bank text-info"></i> Net Banking
                                        </label>
                                    </div>
                                    <div class="form-check">
                                        <input class="form-check-input" type="radio" name="payment_method" id="wallet" value="wallet">
                                        <label class="form-check-label" for="wallet">
                                            <i class="bi bi-wallet2 text-warning"></i> Wallets
                                        </label>
                                    </div>
                                </div>
                                
                                <button class="btn btn-primary btn-lg w-100 mt-3" onclick="initiatePayment()">
                                    <i class="bi bi-lock"></i> Pay ₹<?php echo number_format($order['total_amount'], 2); ?>
                                </button>
                                
                                <div class="text-center mt-3">
                                    <small class="text-muted">
                                        <i class="bi bi-shield-check"></i> Your payment is secured by Razorpay
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="text-center mt-3">
                    <a href="cart.php" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-left"></i> Back to Cart
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <?php include 'includes/footer.php'; ?>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js" integrity="sha256-/xUj+3OJU5yExlq6GSYGSHk7tPXikynS7ogEvDej/m4=" crossorigin="anonymous"></script>
    
    <script>
        function initiatePayment() {
            <?php if (RAZORPAY_TEST_MODE): ?>
            // Test mode - simulate successful payment
            alert('Test Mode: Simulating successful payment...');
            setTimeout(function() {
                // Simulate payment success
                const mockResponse = {
                    razorpay_payment_id: 'pay_test_' + Date.now(),
                    razorpay_order_id: '<?php echo $razorpayOrder['id']; ?>',
                    razorpay_signature: 'test_signature_' + Date.now()
                };
                
                // Submit the mock response
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = 'process_payment.php?order_id=<?php echo $orderId; ?>';
                
                Object.keys(mockResponse).forEach(key => {
                    const input = document.createElement('input');
                    input.type = 'hidden';
                    input.name = key;
                    input.value = mockResponse[key];
                    form.appendChild(input);
                });
                
                document.body.appendChild(form);
                form.submit();
            }, 2000);
            return;
            <?php endif; ?>
            
            const options = {
                key: '<?php echo $razorpay_key_id; ?>',
                amount: <?php echo $order['total_amount'] * 100; ?>,
                currency: 'INR',
                name: 'PoojaFestival.in',
                description: 'Order #<?php echo $orderId; ?>',
                order_id: '<?php echo $razorpayOrder['id']; ?>',
                handler: function(response) {
                    // Handle successful payment
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.action = 'process_payment.php?order_id=<?php echo $orderId; ?>';
                    
                    const fields = {
                        'razorpay_payment_id': response.razorpay_payment_id,
                        'razorpay_order_id': response.razorpay_order_id,
                        'razorpay_signature': response.razorpay_signature
                    };
                    
                    for (const [key, value] of Object.entries(fields)) {
                        const input = document.createElement('input');
                        input.type = 'hidden';
                        input.name = key;
                        input.value = value;
                        form.appendChild(input);
                    }
                    
                    document.body.appendChild(form);
                    form.submit();
                },
                prefill: {
                    name: '<?php echo htmlspecialchars(json_decode($order['shipping_address'], true)['name'] ?? ''); ?>',
                    email: '<?php echo htmlspecialchars(json_decode($order['shipping_address'], true)['email'] ?? ''); ?>',
                    contact: '<?php echo htmlspecialchars(json_decode($order['shipping_address'], true)['phone'] ?? ''); ?>'
                },
                theme: {
                    color: '#ff6b6b'
                },
                modal: {
                    ondismiss: function() {
                        console.log('Payment modal closed');
                    }
                }
            };
            
            const rzp = new Razorpay(options);
            rzp.open();
        }
        
        // Auto-initiate payment after 2 seconds
        setTimeout(function() {
            // Uncomment the line below to auto-start payment
            // initiatePayment();
        }, 2000);
    </script>
</body>
</html> 