<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/Product.php';

class Cart {
    private $items = [];

    public function __construct() {
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
        $this->items = isset($_SESSION['cart']) ? $_SESSION['cart'] : [];
        error_log("Cart initialized. Session ID: " . session_id() . ", Items: " . count($this->items));
    }

    public function addItem($product_id, $quantity = 1) {
        error_log("Adding item to cart: product_id=$product_id, quantity=$quantity");
        
        // Check if Product class is available
        if (!class_exists('Product')) {
            error_log("Product class not found");
            return false;
        }
        
        try {
            $product = Product::getProductById($product_id);
            if (!$product) {
                error_log("Product not found: product_id=$product_id");
                return false;
            }
            
            error_log("Product found: " . print_r($product, true));
        } catch (Exception $e) {
            error_log("Error getting product: " . $e->getMessage());
            return false;
        }

        if (isset($this->items[$product_id])) {
            $this->items[$product_id]['quantity'] += $quantity;
        } else {
            $this->items[$product_id] = [
                'id' => $product_id,
                'name' => $product['name'],
                'price' => $product['sale_price'] ?? $product['price'],
                'quantity' => $quantity,
                'image_url' => $product['image_url']
            ];
        }

        $this->saveCart();
        return true;
    }

    public function updateQuantity($product_id, $quantity) {
        if ($quantity <= 0) {
            unset($this->items[$product_id]);
        } else {
            $this->items[$product_id]['quantity'] = $quantity;
        }
        $this->saveCart();
    }

    public function removeItem($product_id) {
        unset($this->items[$product_id]);
        $this->saveCart();
    }

    public function clear() {
        $this->items = [];
        $this->saveCart();
    }

    public function getItems() {
        return $this->items;
    }

    public function getTotal() {
        $total = 0;
        foreach ($this->items as $item) {
            $total += $item['price'] * $item['quantity'];
        }
        return $total;
    }

    public function getItemCount() {
        $count = 0;
        foreach ($this->items as $item) {
            $count += $item['quantity'];
        }
        return $count;
    }

    private function saveCart() {
        $_SESSION['cart'] = $this->items;
        error_log("Cart saved. Items: " . print_r($this->items, true));
    }
}