# Razorpay Payment Integration Setup Guide

This guide will help you set up Razorpay payment gateway integration for your Pooja Festival e-commerce website.

## Prerequisites

1. **Razorpay Account**: Sign up at [https://razorpay.com](https://razorpay.com)
2. **Domain**: Your website should have a valid domain (for webhooks)
3. **SSL Certificate**: HTTPS is required for payment processing

## Step 1: Get Razorpay API Keys

1. Log in to your Razorpay Dashboard
2. Go to **Settings** → **API Keys**
3. Generate a new key pair
4. Copy the **Key ID** and **Key Secret**

## Step 2: Configure API Keys

1. Open `config/razorpay_config.php`
2. Replace the placeholder values:

```php
// Test Environment
define('RAZORPAY_TEST_KEY_ID', 'rzp_test_YOUR_ACTUAL_TEST_KEY_ID');
define('RAZORPAY_TEST_KEY_SECRET', 'YOUR_ACTUAL_TEST_KEY_SECRET');

// Production Environment
define('RAZORPAY_LIVE_KEY_ID', 'rzp_live_YOUR_ACTUAL_LIVE_KEY_ID');
define('RAZORPAY_LIVE_KEY_SECRET', 'YOUR_ACTUAL_LIVE_KEY_SECRET');
```

## Step 3: Set Up Database

1. Import the orders schema:
```sql
-- Run the SQL from database/orders_schema.sql
```

2. Or execute this command:
```bash
mysql -u your_username -p your_database < database/orders_schema.sql
```

## Step 4: Configure Webhooks (Optional but Recommended)

1. In Razorpay Dashboard, go to **Settings** → **Webhooks**
2. Add a new webhook with URL: `https://yourdomain.com/webhook.php`
3. Select events: `payment.captured`, `payment.failed`
4. Copy the webhook secret and update in `config/razorpay_config.php`:

```php
define('RAZORPAY_WEBHOOK_SECRET', 'YOUR_WEBHOOK_SECRET');
```

## Step 5: Test the Integration

### Test Environment
1. Set `RAZORPAY_ENVIRONMENT` to `'test'` in `config/razorpay_config.php`
2. Use test card details:
   - **Card Number**: 4111 1111 1111 1111
   - **Expiry**: Any future date
   - **CVV**: Any 3 digits
   - **Name**: Any name
   - **3D Secure Password**: 1221

### Test UPI
- **UPI ID**: success@razorpay

## Step 6: Go Live

1. Change `RAZORPAY_ENVIRONMENT` to `'production'`
2. Update with your live API keys
3. Test with real payment methods

## File Structure

```
├── checkout.php              # Main checkout page
├── process_payment.php       # Razorpay payment processing
├── payment_success.php       # Payment success page
├── payment_failed.php        # Payment failure page
├── config/
│   └── razorpay_config.php   # Razorpay configuration
├── classes/
│   └── Order.php            # Order management class
└── database/
    └── orders_schema.sql    # Database schema
```

## Payment Flow

1. **User adds items to cart** → `cart.php`
2. **User proceeds to checkout** → `checkout.php`
3. **User fills shipping details** → `checkout.php`
4. **User selects payment method** → `checkout.php`
5. **Payment processing** → `process_payment.php`
6. **Payment success** → `payment_success.php`
7. **Payment failure** → `payment_failed.php`

## Features

### ✅ Implemented
- [x] Razorpay payment gateway integration
- [x] Multiple payment methods (Card, UPI, Net Banking, Wallets)
- [x] Order management system
- [x] Payment success/failure handling
- [x] Order tracking
- [x] Secure payment verification
- [x] Responsive design
- [x] Form validation
- [x] Error handling

### 🔄 Payment Methods Supported
- **Credit/Debit Cards**: Visa, MasterCard, RuPay, American Express
- **UPI**: All major UPI apps (Google Pay, PhonePe, Paytm, etc.)
- **Net Banking**: 50+ banks
- **Wallets**: Paytm, Mobikwik, Freecharge, etc.
- **EMI**: Available on eligible cards

## Security Features

1. **Payment Signature Verification**: All payments are verified using HMAC SHA256
2. **Order Validation**: Orders are validated against user ownership
3. **Input Sanitization**: All user inputs are properly sanitized
4. **HTTPS Required**: Payment pages require secure connection
5. **Session Management**: Secure session handling

## Error Handling

The system handles various error scenarios:
- Invalid payment details
- Insufficient funds
- Network issues
- Bank declines
- Expired cards
- Invalid UPI IDs

## Customization

### Styling
- Modify `assets/css/style.css` for custom styling
- Update Bootstrap classes in payment pages
- Customize payment button appearance

### Payment Options
- Add/remove payment methods in `config/razorpay_config.php`
- Customize payment flow in `process_payment.php`

### Order Management
- Extend `classes/Order.php` for additional features
- Add order status tracking
- Implement order cancellation

## Troubleshooting

### Common Issues

1. **Payment Not Processing**
   - Check API keys are correct
   - Verify HTTPS is enabled
   - Check browser console for errors

2. **Payment Verification Fails**
   - Ensure webhook secret is correct
   - Check server logs for errors
   - Verify payment signature

3. **Orders Not Creating**
   - Check database connection
   - Verify table structure
   - Check PHP error logs

### Debug Mode

Enable debug mode by adding this to your PHP files:
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

## Support

For technical support:
- **Email**: support@poojafestival.in
- **Phone**: +91 98765 43210
- **Documentation**: [Razorpay Docs](https://razorpay.com/docs/)

## License

This integration is part of the Pooja Festival e-commerce platform.

---

**Note**: Always test thoroughly in test environment before going live. Never commit API keys to version control. 