<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/classes/User.php';
require_once __DIR__ . '/classes/Order.php';

// Start session
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php?redirect=order_details.php');
    exit;
}

// Check if order ID is provided
if (!isset($_GET['id'])) {
    header('Location: profile.php');
    exit;
}

$orderId = (int)$_GET['id'];
$userId = $_SESSION['user_id'];

// Get order with items
$order = Order::getOrderWithItems($orderId);

if (!$order) {
    header('Location: profile.php');
    exit;
}

// Verify order belongs to user
if (!Order::validateOrderOwnership($orderId, $userId)) {
    header('Location: profile.php');
    exit;
}

$shippingAddress = json_decode($order['shipping_address'], true);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order #<?php echo $orderId; ?> - PoojaFestival.in</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.7.2/font/bootstrap-icons.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <!-- Navigation -->
    <?php include 'includes/nav.php'; ?>

    <div class="container my-5">
        <div class="row">
            <div class="col-12">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="profile.php">My Profile</a></li>
                        <li class="breadcrumb-item"><a href="profile.php#orders">My Orders</a></li>
                        <li class="breadcrumb-item active">Order #<?php echo $orderId; ?></li>
                    </ol>
                </nav>
            </div>
        </div>

        <div class="row">
            <div class="col-lg-8">
                <!-- Order Items -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-box"></i> Order Items</h5>
                    </div>
                    <div class="card-body">
                        <?php foreach ($order['items'] as $item): ?>
                        <div class="row mb-3 align-items-center">
                            <div class="col-md-2">
                                <img src="assets/images/<?php echo htmlspecialchars($item['image_url']); ?>" 
                                     class="img-fluid rounded" 
                                     alt="<?php echo htmlspecialchars($item['name']); ?>"
                                     onerror="this.src='assets/images/placeholder.jpg';">
                            </div>
                            <div class="col-md-6">
                                <h6 class="mb-1"><?php echo htmlspecialchars($item['name']); ?></h6>
                                <p class="text-muted mb-0">Quantity: <?php echo $item['quantity']; ?></p>
                            </div>
                            <div class="col-md-2 text-end">
                                <span class="text-muted">₹<?php echo number_format($item['price'], 2); ?> each</span>
                            </div>
                            <div class="col-md-2 text-end">
                                <strong>₹<?php echo number_format($item['price'] * $item['quantity'], 2); ?></strong>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <!-- Order Timeline -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-clock-history"></i> Order Timeline</h5>
                    </div>
                    <div class="card-body">
                        <div class="timeline">
                            <div class="timeline-item">
                                <div class="timeline-marker bg-success"></div>
                                <div class="timeline-content">
                                    <h6>Order Placed</h6>
                                    <p class="text-muted"><?php echo date('M d, Y \a\t g:i A', strtotime($order['created_at'])); ?></p>
                                </div>
                            </div>
                            
                            <?php if ($order['status'] !== 'pending'): ?>
                            <div class="timeline-item">
                                <div class="timeline-marker bg-info"></div>
                                <div class="timeline-content">
                                    <h6>Order Confirmed</h6>
                                    <p class="text-muted">Order has been confirmed and is being processed</p>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (in_array($order['status'], ['shipped', 'delivered'])): ?>
                            <div class="timeline-item">
                                <div class="timeline-marker bg-primary"></div>
                                <div class="timeline-content">
                                    <h6>Order Shipped</h6>
                                    <p class="text-muted">Your order has been shipped and is on its way</p>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if ($order['status'] === 'delivered'): ?>
                            <div class="timeline-item">
                                <div class="timeline-marker bg-success"></div>
                                <div class="timeline-content">
                                    <h6>Order Delivered</h6>
                                    <p class="text-muted">Your order has been successfully delivered</p>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-4">
                <!-- Order Summary -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-receipt"></i> Order Summary</h5>
                    </div>
                    <div class="card-body">
                        <div class="d-flex justify-content-between mb-2">
                            <span>Order ID:</span>
                            <strong>#<?php echo $orderId; ?></strong>
                        </div>
                        <div class="d-flex justify-content-between mb-2">
                            <span>Order Date:</span>
                            <span><?php echo date('M d, Y', strtotime($order['created_at'])); ?></span>
                        </div>
                        <div class="d-flex justify-content-between mb-2">
                            <span>Payment Method:</span>
                            <span class="text-capitalize"><?php echo $order['payment_method']; ?></span>
                        </div>
                        <hr>
                        <div class="d-flex justify-content-between mb-2">
                            <span>Subtotal:</span>
                            <span>₹<?php echo number_format($order['total_amount'], 2); ?></span>
                        </div>
                        <div class="d-flex justify-content-between mb-2">
                            <span>Shipping:</span>
                            <span class="text-success">Free</span>
                        </div>
                        <hr>
                        <div class="d-flex justify-content-between mb-3">
                            <strong>Total:</strong>
                            <strong class="text-primary">₹<?php echo number_format($order['total_amount'], 2); ?></strong>
                        </div>
                        
                        <!-- Status Badges -->
                        <div class="mb-3">
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <span>Order Status:</span>
                                <?php
                                $statusClass = '';
                                switch ($order['status']) {
                                    case 'pending': $statusClass = 'warning'; break;
                                    case 'processing': $statusClass = 'info'; break;
                                    case 'shipped': $statusClass = 'primary'; break;
                                    case 'delivered': $statusClass = 'success'; break;
                                    case 'cancelled': $statusClass = 'danger'; break;
                                }
                                ?>
                                <span class="badge bg-<?php echo $statusClass; ?>">
                                    <?php echo ucfirst($order['status']); ?>
                                </span>
                            </div>
                            <div class="d-flex justify-content-between align-items-center">
                                <span>Payment Status:</span>
                                <?php
                                $paymentClass = $order['payment_status'] === 'paid' ? 'success' : 'warning';
                                ?>
                                <span class="badge bg-<?php echo $paymentClass; ?>">
                                    <?php echo ucfirst($order['payment_status']); ?>
                                </span>
                            </div>
                        </div>
                        
                        <!-- Action Buttons -->
                        <div class="d-grid gap-2">
                            <?php if ($order['status'] === 'pending'): ?>
                            <button class="btn btn-outline-danger" onclick="cancelOrder(<?php echo $orderId; ?>)">
                                <i class="bi bi-x-circle"></i> Cancel Order
                            </button>
                            <?php endif; ?>
                            
                            <a href="profile.php#orders" class="btn btn-outline-secondary">
                                <i class="bi bi-arrow-left"></i> Back to Orders
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Shipping Information -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-truck"></i> Shipping Information</h5>
                    </div>
                    <div class="card-body">
                        <?php if ($shippingAddress): ?>
                        <div>
                            <strong><?php echo htmlspecialchars($shippingAddress['name']); ?></strong><br>
                            <?php echo htmlspecialchars($shippingAddress['address']); ?><br>
                            <?php echo htmlspecialchars($shippingAddress['city'] . ', ' . $shippingAddress['state'] . ' - ' . $shippingAddress['pincode']); ?><br>
                            <strong>Phone:</strong> <?php echo htmlspecialchars($shippingAddress['phone']); ?><br>
                            <strong>Email:</strong> <?php echo htmlspecialchars($shippingAddress['email']); ?>
                        </div>
                        <?php else: ?>
                        <p class="text-muted">No shipping information available.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <?php include 'includes/footer.php'; ?>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js" integrity="sha256-/xUj+3OJU5yExlq6GSYGSHk7tPXikynS7ogEvDej/m4=" crossorigin="anonymous"></script>
    
    <script>
        function cancelOrder(orderId) {
            if (confirm('Are you sure you want to cancel this order? This action cannot be undone.')) {
                // Add AJAX call to cancel order
                $.post('cancel_order.php', {
                    order_id: orderId
                }).then(response => {
                    if (response.success) {
                        alert('Order cancelled successfully!');
                        location.reload();
                    } else {
                        alert('Failed to cancel order: ' + response.message);
                    }
                }).catch(error => {
                    alert('Error cancelling order: ' + error);
                });
            }
        }
    </script>
    
    <style>
        .timeline {
            position: relative;
            padding-left: 30px;
        }
        
        .timeline-item {
            position: relative;
            margin-bottom: 30px;
        }
        
        .timeline-marker {
            position: absolute;
            left: -35px;
            top: 0;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            border: 2px solid #fff;
            box-shadow: 0 0 0 3px #dee2e6;
        }
        
        .timeline-item:not(:last-child)::after {
            content: '';
            position: absolute;
            left: -29px;
            top: 12px;
            width: 2px;
            height: 30px;
            background-color: #dee2e6;
        }
        
        .timeline-content h6 {
            margin-bottom: 5px;
            font-weight: 600;
        }
        
        .timeline-content p {
            margin-bottom: 0;
            font-size: 0.9rem;
        }
    </style>
</body>
</html> 