-- Create database if not exists
CREATE DATABASE IF NOT EXISTS pooja_festival CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

USE pooja_festival;

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    phone VARCHAR(15),
    address TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Categories table
CREATE TABLE IF NOT EXISTS categories (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    slug VARCHAR(100) NOT NULL UNIQUE,
    description TEXT,
    image_url VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Products table
CREATE TABLE IF NOT EXISTS products (
    id INT PRIMARY KEY AUTO_INCREMENT,
    category_id INT,
    name VARCHAR(255) NOT NULL,
    slug VARCHAR(255) NOT NULL UNIQUE,
    description TEXT,
    price DECIMAL(10,2) NOT NULL,
    sale_price DECIMAL(10,2),
    stock INT DEFAULT 0,
    image_url VARCHAR(255),
    is_featured BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE SET NULL
);

-- Product materials table
CREATE TABLE IF NOT EXISTS product_materials (
    id INT PRIMARY KEY AUTO_INCREMENT,
    product_id INT NOT NULL,
    name VARCHAR(100) NOT NULL,
    quantity VARCHAR(50),
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Orders table
CREATE TABLE IF NOT EXISTS orders (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT,
    total_amount DECIMAL(10,2) NOT NULL,
    status ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled') DEFAULT 'pending',
    shipping_address TEXT NOT NULL,
    payment_method VARCHAR(50) NOT NULL,
    payment_status ENUM('pending', 'paid', 'failed') DEFAULT 'pending',
    razorpay_order_id VARCHAR(100),
    razorpay_payment_id VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

-- Order items table
CREATE TABLE IF NOT EXISTS order_items (
    id INT PRIMARY KEY AUTO_INCREMENT,
    order_id INT NOT NULL,
    product_id INT NOT NULL,
    quantity INT NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Custom kit templates table
CREATE TABLE IF NOT EXISTS custom_kit_templates (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    base_price DECIMAL(10,2) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Custom kit items table
CREATE TABLE IF NOT EXISTS custom_kit_items (
    id INT PRIMARY KEY AUTO_INCREMENT,
    template_id INT NOT NULL,
    name VARCHAR(100) NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    is_optional BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (template_id) REFERENCES custom_kit_templates(id) ON DELETE CASCADE
);

-- Subscriptions table
CREATE TABLE IF NOT EXISTS subscriptions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    product_id INT NOT NULL,
    frequency ENUM('weekly', 'monthly', 'quarterly') NOT NULL,
    next_delivery_date DATE NOT NULL,
    status ENUM('active', 'paused', 'cancelled') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Reviews table
CREATE TABLE IF NOT EXISTS reviews (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    product_id INT NOT NULL,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    comment TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE CASCADE
);

-- Insert sample categories
INSERT INTO categories (name, slug, description) VALUES
('Daily Pooja', 'daily-pooja', 'Essential items for daily worship'),
('Festival Kits', 'festival-kits', 'Complete kits for major festivals'),
('Deity Specific', 'deity-specific', 'Items specific to different deities');

-- Insert sample products
INSERT INTO products (category_id, name, slug, description, price, is_featured) VALUES
(1, 'Daily Pooja Kit', 'daily-pooja-kit', 'Essential items including Diya, Incense Sticks, Camphor, Flowers, Haldi-Kumkum, Akshata, Bell, Coconut, Fruits, and Betel Leaves & Nuts', 999.00, TRUE),
(2, 'Ganesh Pooja Kit', 'ganesh-pooja-kit', 'Complete kit with Ganesh Idol, Modak, Durva Grass, Red Flowers, Sandalwood Paste, Coconut, Banana, and Panchamrit', 1499.00, TRUE),
(2, 'Satyanarayan Kit', 'satyanarayan-kit', 'Premium kit containing Katha Book, Panchamrit, Banana Leaves, Fruits, Tulsi Leaves, Dry Fruits, Kalash with accessories', 1999.00, TRUE),
(2, 'Lakshmi Pooja Kit', 'lakshmi-pooja-kit', 'Deluxe kit with Lakshmi-Ganesh Idol, Silver Coin, Lotus Flower, Kumkum, Haldi, Dry Fruits, Sweets, and 21 Diyas', 2499.00, TRUE),
(2, 'Navratri Special Kit', 'navratri-special-kit', 'Complete 9-day kit including Durga Idol, Kalash, Navadhanya, Red Chunri, Ghee Lamp, Coconut, and Durga Saptashati Book', 2999.00, TRUE),
(3, 'Shiv Pooja Kit', 'shiv-pooja-kit', 'Sacred kit with Shiva Lingam, Bilva Leaves, Abhishek Items, White Flowers, Rudraksha Mala, and Bhasm', 1799.00, TRUE),
(3, 'Hanuman Pooja Kit', 'hanuman-pooja-kit', 'Complete kit containing Hanuman Idol, Sindoor, Red Flowers, Laddoo, Coconut, Banana, and Hanuman Chalisa Book', 1299.00, TRUE),
(2, 'Durga Pooja Kit', 'durga-pooja-kit', 'Traditional Bengali style kit with Durga Idol, Red Hibiscus, Sindoor, Fruits, Sweets, and Dhunuchi', 2799.00, TRUE),
(3, 'Krishna Pooja Kit', 'krishna-pooja-kit', 'Divine kit including Krishna-Balaram Idol, Makhan-Mishri, Tulsi Leaves, Flute, Peacock Feather, and Panchamrit', 1699.00, TRUE),
(2, 'Saraswati Pooja Kit', 'saraswati-pooja-kit', 'Blessing kit with Saraswati Idol, White Flowers, Books, Musical Instruments, Kumkum, Haldi, and Yellow Cloth', 1899.00, TRUE),
(2, 'Griha Pravesh Kit', 'griha-pravesh-kit', 'House warming kit containing Kalash, Mango Leaves, Rice, Kumkum, Haldi, Coconut, and Navagraha Samagri', 2299.00, TRUE),
(3, 'Rudrabhishek Kit', 'rudrabhishek-kit', 'Special Shiva kit with Lingam, Panchamrit, Ganga Jal, Bilva Leaves, Dhatura, Black Til, and Rudraksha Mala', 2199.00, TRUE),
(3, 'Sunderkand Path Kit', 'sunderkand-path-kit', 'Complete kit with Hanuman Idol, Red Cloth, Sindoor, Sunderkand Book, Laddoo, and Fruits', 1599.00, TRUE),
(2, 'Ganpati Homam Kit', 'ganpati-homam-kit', 'Special ritual kit containing Ganesh Idol, Modak, Coconut, Havan Samagri, and Red Flowers', 1999.00, TRUE),
(2, 'Satyanarayan Swamy Kit', 'satyanarayan-swamy-kit', 'Premium kit with Photo/Idol, Panchamrit, Fruits, Dry Fruits, Betel Nut, and Kalash set', 2099.00, TRUE);

-- Insert product materials
INSERT INTO product_materials (product_id, name, quantity) VALUES
-- Daily Pooja Kit
(1, 'Brass Diya', '1 piece'),
(1, 'Incense Sticks', '1 pack'),
(1, 'Camphor', '1 box'),
(1, 'Fresh Flowers', '1 bunch'),
(1, 'Haldi-Kumkum Set', '1 set'),
(1, 'Akshata (Unbroken Rice)', '100g'),
(1, 'Brass Bell', '1 piece'),
(1, 'Coconut', '1 piece'),
(1, 'Mixed Fruits', '500g'),
(1, 'Betel Leaves & Nuts', '10 sets'),

-- Ganesh Pooja Kit
(2, 'Clay Ganesh Idol', '1 piece'),
(2, 'Modak', '21 pieces'),
(2, 'Durva Grass', '21 stalks'),
(2, 'Red Hibiscus', '11 flowers'),
(2, 'Sandalwood Paste', '50g'),
(2, 'Coconut', '1 piece'),
(2, 'Banana', '12 pieces'),
(2, 'Panchamrit Set', '1 set'),

-- Satyanarayan Kit
(3, 'Satyanarayan Katha Book', '1 piece'),
(3, 'Panchamrit Set', '1 set'),
(3, 'Banana Leaves', '5 pieces'),
(3, 'Mixed Fruits', '1 kg'),
(3, 'Tulsi Leaves', '108 leaves'),
(3, 'Mixed Dry Fruits', '500g'),
(3, 'Copper Kalash Set', '1 set'),

-- Lakshmi Pooja Kit
(4, 'Lakshmi-Ganesh Idol', '1 set'),
(4, 'Silver Coin', '1 piece'),
(4, 'Lotus Flowers', '5 pieces'),
(4, 'Kumkum', '50g'),
(4, 'Haldi', '50g'),
(4, 'Mixed Dry Fruits', '500g'),
(4, 'Laddoo', '500g'),
(4, 'Brass Diyas', '21 pieces'),

-- Navratri Special Kit
(5, 'Durga Idol', '1 piece'),
(5, 'Brass Kalash Set', '1 set'),
(5, 'Navadhanya (9 Grains)', '9 x 50g'),
(5, 'Red Chunri', '1 piece'),
(5, 'Ghee Lamp', '1 piece'),
(5, 'Coconut', '1 piece'),
(5, 'Durga Saptashati Book', '1 piece'),

-- Shiv Pooja Kit
(6, 'Shiva Lingam', '1 piece'),
(6, 'Bilva Leaves', '108 leaves'),
(6, 'Milk, Honey, Curd Set', '1 set'),
(6, 'White Flowers', '1 bunch'),
(6, 'Rudraksha Mala', '1 piece'),
(6, 'Sacred Ash (Bhasm)', '100g'),

-- Hanuman Pooja Kit
(7, 'Brass Hanuman Idol', '1 piece'),
(7, 'Sindoor', '100g'),
(7, 'Red Flowers', '1 bunch'),
(7, 'Laddoo', '500g'),
(7, 'Coconut', '1 piece'),
(7, 'Banana', '12 pieces'),
(7, 'Hanuman Chalisa Book', '1 piece'),

-- Durga Pooja Kit
(8, 'Durga Idol with Lion', '1 set'),
(8, 'Red Hibiscus', '108 flowers'),
(8, 'Sindoor', '200g'),
(8, 'Mixed Fruits', '2 kg'),
(8, 'Khichdi Prasad', '1 kg'),
(8, 'Dhunuchi', '1 piece'),

-- Krishna Pooja Kit
(9, 'Krishna-Balaram Idol', '1 set'),
(9, 'Makhan-Mishri', '250g'),
(9, 'Tulsi Leaves', '108 leaves'),
(9, 'Decorative Flute', '1 piece'),
(9, 'Peacock Feather', '1 piece'),
(9, 'Panchamrit Set', '1 set'),

-- Saraswati Pooja Kit
(10, 'Saraswati Idol', '1 piece'),
(10, 'White Flowers', '2 bunches'),
(10, 'Blank Notebook', '1 piece'),
(10, 'Musical Bell', '1 piece'),
(10, 'Kumkum-Haldi Set', '1 set'),
(10, 'Yellow Cloth', '1 piece'),

-- Griha Pravesh Kit
(11, 'Brass Kalash Set', '1 set'),
(11, 'Mango Leaves', '21 leaves'),
(11, 'Sacred Rice', '250g'),
(11, 'Kumkum-Haldi Set', '1 set'),
(11, 'Coconut', '1 piece'),
(11, 'Navagraha Samagri', '9 sets'),

-- Rudrabhishek Kit
(12, 'Shiva Lingam', '1 piece'),
(12, 'Panchamrit Set', '1 set'),
(12, 'Ganga Jal', '200ml'),
(12, 'Bilva Leaves', '108 leaves'),
(12, 'Dhatura Flowers', '11 pieces'),
(12, 'Black Til', '100g'),
(12, 'Rudraksha Mala', '1 piece'),

-- Sunderkand Path Kit
(13, 'Brass Hanuman Idol', '1 piece'),
(13, 'Red Cotton Cloth', '1 piece'),
(13, 'Sindoor', '100g'),
(13, 'Sunderkand Book', '1 piece'),
(13, 'Laddoo Prasad', '500g'),
(13, 'Mixed Fruits', '1 kg'),

-- Ganpati Homam Kit
(14, 'Clay Ganesh Idol', '1 piece'),
(14, 'Modak', '21 pieces'),
(14, 'Coconut', '2 pieces'),
(14, 'Havan Samagri', '500g'),
(14, 'Red Flowers', '2 bunches'),

-- Satyanarayan Swamy Kit
(15, 'Framed Photo', '1 piece'),
(15, 'Panchamrit Set', '1 set'),
(15, 'Mixed Fruits', '2 kg'),
(15, 'Mixed Dry Fruits', '500g'),
(15, 'Betel Nut Set', '100g'),
(15, 'Brass Kalash Set', '1 set');