<?php
require_once __DIR__ . '/../config/database.php';

class User {
    private $id;
    private $name;
    private $email;
    private $password;
    private $phone;
    private $address;
    private $role;
    private $created_at;
    private $updated_at;

    // Create new user
    public static function createUser($data) {
        $conn = connectDB();
        if (!$conn) {
            return false;
        }

        $name = $conn->real_escape_string($data['name']);
        $email = $conn->real_escape_string($data['email']);
        $password = password_hash($data['password'], PASSWORD_DEFAULT);
        $phone = isset($data['phone']) ? "'" . $conn->real_escape_string($data['phone']) . "'" : 'NULL';
        $address = isset($data['address']) ? "'" . $conn->real_escape_string($data['address']) . "'" : 'NULL';
        $role = isset($data['role']) ? "'" . $conn->real_escape_string($data['role']) . "'" : "'user'";

        $query = "INSERT INTO users (name, email, password, phone, address, role) 
                  VALUES ('$name', '$email', '$password', $phone, $address, $role)";

        if ($conn->query($query)) {
            $userId = $conn->insert_id;
            closeDB($conn);
            return $userId;
        }

        closeDB($conn);
        return false;
    }

    // Get user by ID
    public static function getUserById($id) {
        $id = (int)$id;
        $query = "SELECT * FROM users WHERE id = $id";
        return getRow($query);
    }

    // Get user by email
    public static function getUserByEmail($email) {
        $email = sanitizeInput($email);
        $query = "SELECT * FROM users WHERE email = '$email'";
        return getRow($query);
    }

    // Update user
    public static function updateUser($id, $data) {
        $conn = connectDB();
        if (!$conn) {
            return false;
        }

        $id = (int)$id;
        $updateFields = [];

        if (isset($data['name'])) {
            $updateFields[] = "name = '" . $conn->real_escape_string($data['name']) . "'";
        }
        if (isset($data['email'])) {
            $updateFields[] = "email = '" . $conn->real_escape_string($data['email']) . "'";
        }
        if (isset($data['password'])) {
            $updateFields[] = "password = '" . $conn->real_escape_string($data['password']) . "'";
        }
        if (isset($data['phone'])) {
            $updateFields[] = "phone = '" . $conn->real_escape_string($data['phone']) . "'";
        }
        if (isset($data['address'])) {
            $updateFields[] = "address = '" . $conn->real_escape_string($data['address']) . "'";
        }
        if (isset($data['role'])) {
            $updateFields[] = "role = '" . $conn->real_escape_string($data['role']) . "'";
        }

        if (empty($updateFields)) {
            closeDB($conn);
            return false;
        }

        $updateFields[] = "updated_at = CURRENT_TIMESTAMP";
        $query = "UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = $id";

        $result = $conn->query($query);
        closeDB($conn);
        return $result;
    }

    // Delete user
    public static function deleteUser($id) {
        $id = (int)$id;
        $query = "DELETE FROM users WHERE id = $id";
        return executeQuery($query);
    }

    // Authenticate user
    public static function authenticate($email, $password) {
        $user = self::getUserByEmail($email);
        
        if ($user && password_verify($password, $user['password'])) {
            return $user;
        }
        
        return false;
    }

    // Check if email exists
    public static function emailExists($email, $excludeId = null) {
        $email = sanitizeInput($email);
        $query = "SELECT id FROM users WHERE email = '$email'";
        
        if ($excludeId) {
            $excludeId = (int)$excludeId;
            $query .= " AND id != $excludeId";
        }
        
        $result = getRow($query);
        return !empty($result);
    }

    // Get all users (for admin)
    public static function getAllUsers($limit = null, $offset = 0) {
        $query = "SELECT id, name, email, phone, role, created_at FROM users ORDER BY created_at DESC";
        
        if ($limit) {
            $query .= " LIMIT " . (int)$limit . " OFFSET " . (int)$offset;
        }
        
        return getRows($query);
    }

    // Get users by role
    public static function getUsersByRole($role) {
        $role = sanitizeInput($role);
        $query = "SELECT * FROM users WHERE role = '$role' ORDER BY created_at DESC";
        return getRows($query);
    }

    // Get total users count
    public static function getTotalUsersCount() {
        $query = "SELECT COUNT(*) as count FROM users";
        $result = getRow($query);
        return $result ? $result['count'] : 0;
    }

    // Get user statistics
    public static function getUserStats() {
        $stats = [];
        
        // Total users
        $stats['total_users'] = self::getTotalUsersCount();
        
        // Users by role
        $stats['admin_users'] = count(self::getUsersByRole('admin'));
        $stats['regular_users'] = count(self::getUsersByRole('user'));
        
        // Recent registrations (last 30 days)
        $query = "SELECT COUNT(*) as count FROM users WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
        $result = getRow($query);
        $stats['recent_registrations'] = $result ? $result['count'] : 0;
        
        return $stats;
    }

    // Change user password
    public static function changePassword($userId, $currentPassword, $newPassword) {
        $user = self::getUserById($userId);
        
        if (!$user || !password_verify($currentPassword, $user['password'])) {
            return false;
        }
        
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        return self::updateUser($userId, ['password' => $hashedPassword]);
    }

    // Reset password (for admin or forgot password)
    public static function resetPassword($userId, $newPassword) {
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        return self::updateUser($userId, ['password' => $hashedPassword]);
    }

    // Get user orders
    public static function getUserOrders($userId) {
        require_once __DIR__ . '/Order.php';
        return Order::getUserOrders($userId);
    }

    // Get user order count
    public static function getUserOrderCount($userId) {
        $userId = (int)$userId;
        $query = "SELECT COUNT(*) as count FROM orders WHERE user_id = $userId";
        $result = getRow($query);
        return $result ? $result['count'] : 0;
    }

    // Get user total spent
    public static function getUserTotalSpent($userId) {
        $userId = (int)$userId;
        $query = "SELECT SUM(total_amount) as total FROM orders WHERE user_id = $userId AND payment_status = 'paid'";
        $result = getRow($query);
        return $result ? $result['total'] : 0;
    }

    // Search users
    public static function searchUsers($searchTerm, $limit = 10) {
        $searchTerm = sanitizeInput($searchTerm);
        $limit = (int)$limit;
        
        $query = "SELECT id, name, email, phone, role, created_at 
                  FROM users 
                  WHERE name LIKE '%$searchTerm%' 
                  OR email LIKE '%$searchTerm%' 
                  OR phone LIKE '%$searchTerm%' 
                  ORDER BY created_at DESC 
                  LIMIT $limit";
        
        return getRows($query);
    }

    // Validate user data
    public static function validateUserData($data, $isUpdate = false) {
        $errors = [];
        
        if (!$isUpdate || isset($data['name'])) {
            if (empty($data['name'])) {
                $errors[] = 'Name is required';
            } elseif (strlen($data['name']) < 2) {
                $errors[] = 'Name must be at least 2 characters long';
            }
        }
        
        if (!$isUpdate || isset($data['email'])) {
            if (empty($data['email'])) {
                $errors[] = 'Email is required';
            } elseif (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'Please enter a valid email address';
            } elseif (self::emailExists($data['email'], $isUpdate ? $data['id'] : null)) {
                $errors[] = 'Email address already exists';
            }
        }
        
        if (!$isUpdate || isset($data['password'])) {
            if (empty($data['password'])) {
                $errors[] = 'Password is required';
            } elseif (strlen($data['password']) < 6) {
                $errors[] = 'Password must be at least 6 characters long';
            }
        }
        
        if (isset($data['phone']) && !empty($data['phone'])) {
            if (!preg_match('/^[0-9]{10}$/', $data['phone'])) {
                $errors[] = 'Please enter a valid 10-digit phone number';
            }
        }
        
        return $errors;
    }

    // Get user activity (recent orders, etc.)
    public static function getUserActivity($userId, $limit = 5) {
        $userId = (int)$userId;
        $limit = (int)$limit;
        
        $query = "SELECT o.*, COUNT(oi.id) as item_count 
                  FROM orders o 
                  LEFT JOIN order_items oi ON o.id = oi.order_id 
                  WHERE o.user_id = $userId 
                  GROUP BY o.id 
                  ORDER BY o.created_at DESC 
                  LIMIT $limit";
        
        return getRows($query);
    }

    // Check if user is admin
    public static function isAdmin($userId) {
        $user = self::getUserById($userId);
        return $user && $user['role'] === 'admin';
    }

    // Get user profile completeness
    public static function getProfileCompleteness($userId) {
        $user = self::getUserById($userId);
        if (!$user) {
            return 0;
        }
        
        $fields = ['name', 'email', 'phone', 'address'];
        $completed = 0;
        
        foreach ($fields as $field) {
            if (!empty($user[$field])) {
                $completed++;
            }
        }
        
        return round(($completed / count($fields)) * 100);
    }
}
?>