<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__. '/Order.php';
require_once __DIR__. '/User.php';

class Payment {
    private static $razorpay_key_id = 'YOUR_RAZORPAY_KEY_ID'; // Replace with actual key in production
    private static $razorpay_key_secret = 'YOUR_RAZORPAY_KEY_SECRET'; // Replace with actual key in production

    // Initialize Razorpay order
    public static function initializePayment($order_id) {
        $order = Order::getOrderById($order_id);
        if (!$order) {
            return ['error' => 'Order not found'];
        }

        try {
            // Initialize Razorpay API
            $api = new Razorpay\Api\Api(self::$razorpay_key_id, self::$razorpay_key_secret);

            // Create Razorpay order
            $razorpay_order = $api->order->create([
                'amount' => $order['total_amount'] * 100, // Amount in paise
                'currency' => 'INR',
                'receipt' => 'order_' . $order_id,
                'payment_capture' => 1
            ]);

            // Update order with Razorpay order ID
            Order::setRazorpayOrderId($order_id, $razorpay_order->id);

            return [
                'success' => true,
                'order_id' => $razorpay_order->id,
                'amount' => $order['total_amount'],
                'currency' => 'INR',
                'key_id' => self::$razorpay_key_id
            ];
        } catch (Exception $e) {
            error_log('Razorpay Error: ' . $e->getMessage());
            return ['error' => 'Payment initialization failed'];
        }
    }

    // Verify payment signature
    public static function verifyPayment($payment_id, $order_id, $signature) {
        try {
            // Get the Razorpay order ID from our database
            $order = Order::getOrderById($order_id);
            if (!$order || !$order['razorpay_order_id']) {
                return ['error' => 'Invalid order'];
            }

            // Verify signature
            $expected_signature = hash_hmac(
                'sha256',
                $order['razorpay_order_id'] . '|' . $payment_id,
                self::$razorpay_key_secret
            );

            if ($expected_signature === $signature) {
                // Update order payment status
                Order::updatePaymentStatus($order_id, 'paid', $payment_id);
                return ['success' => true];
            }

            return ['error' => 'Invalid payment signature'];
        } catch (Exception $e) {
            error_log('Payment Verification Error: ' . $e->getMessage());
            return ['error' => 'Payment verification failed'];
        }
    }

    // Handle payment success
    public static function handlePaymentSuccess($order_id, $payment_id, $signature) {
        // Verify payment first
        $verification = self::verifyPayment($payment_id, $order_id, $signature);
        if (isset($verification['error'])) {
            return $verification;
        }

        try {
            // Update order status
            Order::updateOrderStatus($order_id, 'processing');

            // Get order details for confirmation
            $order = Order::getOrderById($order_id);
            $items = Order::getOrderItems($order_id);

            // Send confirmation email (implement this based on your needs)
            self::sendPaymentConfirmation($order, $items);

            return [
                'success' => true,
                'message' => 'Payment processed successfully',
                'order_id' => $order_id
            ];
        } catch (Exception $e) {
            error_log('Payment Success Handler Error: ' . $e->getMessage());
            return ['error' => 'Error processing payment success'];
        }
    }

    // Handle payment failure
    public static function handlePaymentFailure($order_id, $error_code = null, $error_description = null) {
        try {
            // Update order payment status
            Order::updatePaymentStatus($order_id, 'failed');

            // Log payment failure
            $error_details = json_encode([
                'order_id' => $order_id,
                'error_code' => $error_code,
                'error_description' => $error_description,
                'timestamp' => date('Y-m-d H:i:s')
            ]);
            error_log('Payment Failure: ' . $error_details);

            return [
                'success' => false,
                'message' => 'Payment failed',
                'order_id' => $order_id
            ];
        } catch (Exception $e) {
            error_log('Payment Failure Handler Error: ' . $e->getMessage());
            return ['error' => 'Error processing payment failure'];
        }
    }

    // Send payment confirmation email
    private static function sendPaymentConfirmation($order, $items) {
        // Get user details
        $user = User::getUserById($order['user_id']);
        if (!$user) {
            return false;
        }

        // Prepare email content
        $to = $user['email'];
        $subject = 'Order Confirmation - PoojaFestival.in';

        $message = "<html><body>";
        $message .= "<h2>Thank you for your order!</h2>";
        $message .= "<p>Order ID: {$order['id']}</p>";
        $message .= "<p>Amount Paid: ₹{$order['total_amount']}</p>";
        $message .= "<h3>Order Items:</h3><ul>";

        foreach ($items as $item) {
            $message .= "<li>{$item['name']} x {$item['quantity']} - ₹{$item['price']}</li>";
        }

        $message .= "</ul>";
        $message .= "<p>Shipping Address: {$order['shipping_address']}</p>";
        $message .= "<p>We will process your order shortly.</p>";
        $message .= "</body></html>";

        // Email headers
        $headers = "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        $headers .= "From: PoojaFestival.in <noreply@poojafestival.in>\r\n";

        // Send email
        return mail($to, $subject, $message, $headers);
    }

    // Get payment status
    public static function getPaymentStatus($payment_id) {
        try {
            $api = new Razorpay\Api\Api(self::$razorpay_key_id, self::$razorpay_key_secret);
            $payment = $api->payment->fetch($payment_id);
            return $payment->status;
        } catch (Exception $e) {
            error_log('Get Payment Status Error: ' . $e->getMessage());
            return false;
        }
    }

    // Refund payment
    public static function refundPayment($payment_id, $amount = null) {
        try {
            $api = new Razorpay\Api\Api(self::$razorpay_key_id, self::$razorpay_key_secret);
            
            $refund_data = [];
            if ($amount) {
                $refund_data['amount'] = $amount * 100; // Convert to paise
            }

            $refund = $api->payment->fetch($payment_id)->refund($refund_data);
            return ['success' => true, 'refund_id' => $refund->id];
        } catch (Exception $e) {
            error_log('Refund Error: ' . $e->getMessage());
            return ['error' => 'Refund failed'];
        }
    }
}
?>