// Cart functionality
class ShoppingCart {
    static addToCart(productId, quantity = 1) {
        console.log('Adding to cart:', { productId, quantity });
        
        // Check if jQuery is available
        if (typeof $ === 'undefined') {
            console.error('jQuery is not loaded');
            this.showNotification('Error: jQuery not loaded', 'danger');
            return Promise.reject(new Error('jQuery not loaded'));
        }
        
        // Log the request details
        console.log('Making AJAX request to cart-actions.php');
        console.log('Request data:', {
            action: 'add',
            product_id: productId,
            quantity: quantity
        });
        
        return $.post('cart-actions.php', {
            action: 'add',
            product_id: productId,
            quantity: quantity
        }).then(response => {
            console.log('Cart response received:', response);
            console.log('Response type:', typeof response);
            console.log('Response success:', response.success);
            
            if (response.success) {
                this.updateCartCount(response.cartCount);
                this.showNotification('Product added to cart successfully!');
                
                // Refresh cart offcanvas if it's open
                this.refreshCartOffcanvas();
            } else {
                console.error('Failed to add to cart:', response);
                this.showNotification('Failed to add product to cart: ' + (response.error || 'Unknown error'), 'danger');
            }
            return response;
        }).catch(error => {
            console.error('Cart request failed:', error);
            console.error('Error details:', {
                status: error.status,
                statusText: error.statusText,
                responseText: error.responseText
            });
            this.showNotification('Network error. Please try again.', 'danger');
            return { success: false, error: error.message };
        });
    }

    static updateCartCount(count) {
        if (typeof $ !== 'undefined') {
            const cartCountElement = $('.cart-count');
            cartCountElement.text(count);
            
            // Show/hide badge based on count
            if (count > 0) {
                cartCountElement.show();
            } else {
                cartCountElement.hide();
            }
            
            // Add animation class
            cartCountElement.addClass('updated');
            setTimeout(() => {
                cartCountElement.removeClass('updated');
            }, 500);
        }
    }

    static refreshCartOffcanvas() {
        if (typeof $ !== 'undefined') {
            // Check if cart offcanvas is open
            const cartOffcanvas = document.getElementById('cartOffcanvas');
            if (cartOffcanvas && cartOffcanvas.classList.contains('show')) {
                // Trigger the show event to reload cart contents
                const event = new Event('show.bs.offcanvas');
                cartOffcanvas.dispatchEvent(event);
            }
        }
    }

    static refreshCartCount() {
        if (typeof $ !== 'undefined') {
            $.post('cart-actions.php', {
                action: 'get'
            }).then(function(response) {
                if (response.success) {
                    ShoppingCart.updateCartCount(response.cartCount);
                }
            }).catch(function(error) {
                console.error('Error refreshing cart count:', error);
            });
        }
    }

    static showNotification(message, type = 'success') {
        console.log('Showing notification:', message, type);
        
        // Check if jQuery is available
        if (typeof $ === 'undefined') {
            console.error('jQuery not available for notification');
            alert(message);
            return;
        }
        
        // Create notification element
        const notification = $(`
            <div class="toast-container position-fixed top-0 end-0 p-3" style="z-index: 9999;">
                <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
                    <div class="d-flex">
                        <div class="toast-body">${message}</div>
                        <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                    </div>
                </div>
            </div>
        `);

        // Remove any existing notifications
        $('.toast-container').remove();
        
        // Add new notification
        $('body').append(notification);
        
        // Show notification
        if (typeof bootstrap !== 'undefined' && bootstrap.Toast) {
            try {
                const toast = new bootstrap.Toast(notification.find('.toast'));
                toast.show();
            } catch (e) {
                console.error('Bootstrap toast error:', e);
                // Fallback to simple alert
                alert(message);
                notification.remove();
            }
        } else {
            // Fallback for when Bootstrap is not available
            console.log('Bootstrap not available, using fallback notification');
            alert(message);
            notification.remove();
        }

        // Remove notification after it's hidden
        notification.on('hidden.bs.toast', function () {
            notification.remove();
        });
    }
}

// Wait for jQuery to be loaded before setting up event handlers
function initializeCartHandlers() {
    if (typeof $ === 'undefined') {
        console.log('jQuery not loaded yet, retrying in 100ms...');
        setTimeout(initializeCartHandlers, 100);
        return;
    }
    
    console.log('Initializing cart handlers with jQuery version:', $.fn.jquery);
    
    // Add to cart button handler
    $(document).on('click', '.add-to-cart', function(e) {
        e.preventDefault();
        console.log('Add to cart button clicked');
        
        const productId = $(this).data('product-id');
        const quantity = $(this).closest('.product-card').find('.quantity-input').val() || 1;
        
        console.log('Product ID:', productId, 'Quantity:', quantity);
        
        if (!productId) {
            console.error('No product ID found on button');
            ShoppingCart.showNotification('Error: Product ID not found', 'danger');
            return;
        }

        ShoppingCart.addToCart(productId, quantity).then(response => {
            console.log('Add to cart completed:', response);
            if (!response.success) {
                ShoppingCart.showNotification('Failed to add product to cart.', 'danger');
            }
        });
    });

    // Quantity input handler
    $(document).on('change', '.quantity-input', function() {
        const value = parseInt($(this).val());
        if (value < 1) {
            $(this).val(1);
        } else if (value > 99) {
            $(this).val(99);
        }
    });

    // Initialize tooltips and popovers
    $(document).ready(function() {
        $('[data-bs-toggle="tooltip"]').tooltip();
        $('[data-bs-toggle="popover"]').popover();
    });
}

// Function to check if all dependencies are loaded
function checkDependencies() {
    if (typeof $ === 'undefined') {
        console.log('jQuery not available yet, waiting...');
        return false;
    }
    
    if (typeof ShoppingCart === 'undefined') {
        console.log('ShoppingCart not available yet, waiting...');
        return false;
    }
    
    console.log('All dependencies loaded successfully');
    return true;
}

// Initialize when everything is ready
function initializeWhenReady() {
    if (checkDependencies()) {
        initializeCartHandlers();
        console.log('Cart system initialized successfully');
    } else {
        setTimeout(initializeWhenReady, 100);
    }
}

// Global function to update cart count (can be called from anywhere)
window.updateCartCount = function(count) {
    ShoppingCart.updateCartCount(count);
};

// Start initialization
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeWhenReady);
} else {
    initializeWhenReady();
}

// Final check after 3 seconds
setTimeout(function() {
    if (!checkDependencies()) {
        console.error('Cart system failed to initialize after 3 seconds');
        console.log('jQuery available:', typeof $ !== 'undefined');
        console.log('ShoppingCart available:', typeof ShoppingCart !== 'undefined');
        
        // Try to load jQuery manually if it's not available
        if (typeof $ === 'undefined') {
            console.log('Attempting to load jQuery manually...');
            const script = document.createElement('script');
            script.src = 'https://code.jquery.com/jquery-3.6.0.min.js';
            script.onload = function() {
                console.log('jQuery loaded manually, reinitializing...');
                initializeWhenReady();
            };
            document.head.appendChild(script);
        }
    }
}, 3000);