<?php
require_once '../includes/auth.php';
require_once '../config/db.php';

// Get student ID from URL
$student_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);

if (!$student_id) {
    header('Location: students.php');
    exit();
}

// Get student data with class information
$sql = "SELECT s.*, c.class_name 
        FROM students s 
        JOIN classes c ON s.class_id = c.class_id 
        WHERE s.student_id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $student_id);
$stmt->execute();
$result = $stmt->get_result();
$student = $result->fetch_assoc();

if (!$student) {
    header('Location: students.php');
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <?php include '../includes/header-internal.php'; ?>
    <title>View Student - <?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></title>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Student Details</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="edit.php?id=<?php echo $student_id; ?>" class="btn btn-sm btn-outline-primary me-2">
                            <i class="fas fa-edit"></i> Edit
                        </a>
                        <a href="students.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-arrow-left"></i> Back to List
                        </a>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="card mb-4">
                            <div class="card-header">
                                <h5 class="card-title mb-0">Personal Information</h5>
                            </div>
                            <div class="card-body">
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Admission Number:</div>
                                    <div class="col-sm-8"><?php echo htmlspecialchars($student['admission_number']); ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Name:</div>
                                    <div class="col-sm-8"><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Class:</div>
                                    <div class="col-sm-8"><?php echo htmlspecialchars($student['class_name']); ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Academic Year:</div>
                                    <div class="col-sm-8"><?php echo htmlspecialchars($student['academic_year']); ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Admission Date:</div>
                                    <div class="col-sm-8"><?php echo date('F j, Y', strtotime($student['admission_date'])); ?></div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Status:</div>
                                    <div class="col-sm-8">
                                        <span class="badge bg-<?php echo $student['status'] == 'active' ? 'success' : 'danger'; ?>">
                                            <?php echo ucfirst($student['status']); ?>
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-6">
                        <div class="card mb-4">
                            <div class="card-header">
                                <h5 class="card-title mb-0">Contact Information</h5>
                            </div>
                            <div class="card-body">
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Contact Number:</div>
                                    <div class="col-sm-8">
                                        <?php echo $student['contact_number'] ? htmlspecialchars($student['contact_number']) : '<span class="text-muted">Not provided</span>'; ?>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Parent/Guardian:</div>
                                    <div class="col-sm-8">
                                        <?php echo $student['parent_name'] ? htmlspecialchars($student['parent_name']) : '<span class="text-muted">Not provided</span>'; ?>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-sm-4 fw-bold">Address:</div>
                                    <div class="col-sm-8">
                                        <?php echo $student['address'] ? nl2br(htmlspecialchars($student['address'])) : '<span class="text-muted">Not provided</span>'; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Fee History Section -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0">Fee History</h5>
                    </div>
                    <div class="card-body">
                        <?php
                        // Get fee history
                        $fee_sql = "SELECT * FROM fee_transactions 
                                  WHERE student_id = ? 
                                  ORDER BY transaction_date DESC";
                        $stmt = $conn->prepare($fee_sql);
                        $stmt->bind_param("i", $student_id);
                        $stmt->execute();
                        $fee_result = $stmt->get_result();
                        
                        if ($fee_result->num_rows > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Receipt No.</th>
                                            <th>Amount Paid</th>
                                            <th>Payment Mode</th>
                                            <th>Balance</th>
                                            <th>Installment</th>
                                            <th>Collected By</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($fee = $fee_result->fetch_assoc()): ?>
                                            <tr>
                                                <td><?php echo date('F j, Y', strtotime($fee['transaction_date'])); ?></td>
                                                <td><?php echo htmlspecialchars($fee['receipt_number']); ?></td>
                                                <td>₹<?php echo number_format($fee['amount_paid'], 2); ?></td>
                                                <td><?php echo htmlspecialchars($fee['payment_mode']); ?></td>
                                                <td>₹<?php echo number_format($fee['balance_amount'], 2); ?></td>
                                                <td><?php echo $fee['installment_number']; ?></td>
                                                <td><?php echo htmlspecialchars($fee['collected_by']); ?></td>
                                            </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="alert alert-info">
                                No fee records found for this student.
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>
</body>
</html> 