<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/db.php';

// Check if user has permission to delete students
if (!hasRole('admin')) {
    die("Access denied. Only administrators can delete students.");
}

// Get student ID
$student_id = isset($_GET['id']) ? intval($_GET['id']) : 0;

if ($student_id <= 0) {
    header("Location: students.php?error=Invalid student ID");
    exit();
}

// Get student details first
$sql = "SELECT s.*, c.class_name 
        FROM students s
        JOIN classes c ON s.class_id = c.class_id
        WHERE s.student_id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param('i', $student_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows == 0) {
    header("Location: students.php?error=Student not found");
    exit();
}

$student = $result->fetch_assoc();

// Check if student has any fee transactions
$sql = "SELECT COUNT(*) as transaction_count FROM fee_transactions WHERE student_id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param('i', $student_id);
$stmt->execute();
$result = $stmt->get_result();
$transaction_count = $result->fetch_assoc()['transaction_count'];

// If student has transactions, we should not delete but deactivate
if ($transaction_count > 0) {
    // Update status to inactive instead of deleting
    $sql = "UPDATE students SET status = 'inactive' WHERE student_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('i', $student_id);
    
    if ($stmt->execute()) {
        header("Location: students.php?success=Student deactivated successfully (cannot delete due to existing transactions)");
    } else {
        header("Location: students.php?error=Failed to deactivate student");
    }
    exit();
}

// If no transactions exist, we can safely delete the student
try {
    // Start transaction
    $conn->begin_transaction();
    
    // Delete related records first (if any exist)
    // Note: These tables might not exist yet, but we'll handle them if they do
    
    // Delete from monthly_payments if table exists
    $sql = "DELETE FROM monthly_payments WHERE student_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('i', $student_id);
    $stmt->execute();
    
    // Finally delete the student
    $sql = "DELETE FROM students WHERE student_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('i', $student_id);
    
    if ($stmt->execute()) {
        $conn->commit();
        header("Location: students.php?success=Student deleted successfully");
    } else {
        throw new Exception("Failed to delete student");
    }
    
} catch (Exception $e) {
    $conn->rollback();
    header("Location: students.php?error=Error deleting student: " . $e->getMessage());
}

exit();
?> 