<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/db.php';

// Set page title
$pageTitle = "Section-wise Collection Report";

// Default date range (current month)
$start_date = date('Y-m-01');
$end_date = date('Y-m-t');

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $start_date = $_POST['start_date'];
    $end_date = $_POST['end_date'];
}

// Get section-wise cash and other totals
$sql = "SELECT 
            sec.section_name,
            SUM(CASE WHEN ft.payment_mode = 'Cash' THEN ft.amount_paid ELSE 0 END) as cash_total,
            SUM(CASE WHEN ft.payment_mode != 'Cash' THEN ft.amount_paid ELSE 0 END) as other_total
        FROM sections sec
        LEFT JOIN classes c ON sec.section_id = c.section_id
        LEFT JOIN students s ON c.class_id = s.class_id AND s.status = 'active'
        LEFT JOIN fee_transactions ft ON s.student_id = ft.student_id 
            AND DATE(ft.transaction_date) BETWEEN ? AND ?
        GROUP BY sec.section_id, sec.section_name
        ORDER BY sec.section_id";

$stmt = $conn->prepare($sql);
$stmt->bind_param('ss', $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();
$section_collections = $result->fetch_all(MYSQLI_ASSOC);

// Calculate grand totals
$grand_cash_total = 0;
$grand_other_total = 0;
foreach ($section_collections as $section) {
    $grand_cash_total += $section['cash_total'];
    $grand_other_total += $section['other_total'];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <?php include '../includes/header-internal.php'; ?>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Section-wise Collection Report</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <button class="btn btn-sm btn-outline-secondary" onclick="window.print()">
                            <i class="fas fa-print"></i> Print Report
                        </button>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <div class="col-md-5">
                                <label for="start_date" class="form-label">From Date</label>
                                <input type="date" class="form-control" id="start_date" name="start_date" 
                                       value="<?= $start_date ?>" required>
                            </div>
                            <div class="col-md-5">
                                <label for="end_date" class="form-label">To Date</label>
                                <input type="date" class="form-control" id="end_date" name="end_date" 
                                       value="<?= $end_date ?>" required>
                            </div>
                            <div class="col-md-2 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary w-100">Generate</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Section-wise Collection (by Payment Mode)</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped">
                                <thead>
                                    <tr>
                                        <th>Section</th>
                                        <th>Cash Total</th>
                                        <th>Bank Total</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($section_collections as $section): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($section['section_name']) ?></td>
                                        <td>₹<?= number_format($section['cash_total'], 2) ?></td>
                                        <td>₹<?= number_format($section['other_total'], 2) ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot>
                                    <tr class="table-active">
                                        <th>Total</th>
                                        <th>₹<?= number_format($grand_cash_total, 2) ?></th>
                                        <th>₹<?= number_format($grand_other_total, 2) ?></th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 