<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/db.php';

// Set page title
$pageTitle = "Student Fee Statement";

// Get all students for dropdown
$sql_students = "SELECT s.student_id, s.admission_number, 
                        CONCAT(s.first_name, ' ', s.last_name) as student_name,
                        c.class_name, sec.section_name
                 FROM students s
                 JOIN classes c ON s.class_id = c.class_id
                 JOIN sections sec ON c.section_id = sec.section_id
                 ORDER BY sec.section_id, c.class_name, s.first_name";
$students_result = $conn->query($sql_students);
$students = $students_result->fetch_all(MYSQLI_ASSOC);

// Handle form submission
$student_id = null;
$student_data = null;
$fee_structure = null;
$transactions = [];
$balance = 0;

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['student_id'])) {
    $student_id = intval($_POST['student_id']);
    
    // Get student data
    $sql = "SELECT s.*, c.class_name, sec.section_name
            FROM students s
            JOIN classes c ON s.class_id = c.class_id
            JOIN sections sec ON c.section_id = sec.section_id
            WHERE s.student_id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('i', $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $student_data = $result->fetch_assoc();
    
    if ($student_data) {
        // Get fee structure for the student's class
        $sql = "SELECT * FROM fee_structure 
                WHERE class_id = ? AND academic_year = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param('is', $student_data['class_id'], $student_data['academic_year']);
        $stmt->execute();
        $result = $stmt->get_result();
        $fee_structure = $result->fetch_assoc();
        
        // Get all transactions for the student
        $sql = "SELECT * FROM fee_transactions
                WHERE student_id = ? AND academic_year = ?
                ORDER BY transaction_date";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param('is', $student_id, $student_data['academic_year']);
        $stmt->execute();
        $result = $stmt->get_result();
        $transactions = $result->fetch_all(MYSQLI_ASSOC);
        
        // Calculate balance
        $total_paid = array_sum(array_column($transactions, 'amount_paid'));
        $balance = $fee_structure ? $fee_structure['total_fee'] - $total_paid : 0;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
      <?php include '../includes/header-internal.php'; ?>
</head>
<body>
 
    <div class="container-fluid">
        <div class="row">
          
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Student Fee Statement</h1>
                    <?php if ($student_data): ?>
                    <button class="btn btn-sm btn-outline-secondary" onclick="window.print()">
                        <i class="fas fa-print"></i> Print Statement
                    </button>
                    <?php endif; ?>
                </div>
                
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <div class="col-md-8">
                                <label for="student_id" class="form-label">Select Student</label>
                                <select class="form-select" id="student_id" name="student_id" required>
                                    <option value="">Select Student</option>
                                    <?php foreach ($students as $student): ?>
                                    <option value="<?= $student['student_id'] ?>" <?= $student_id == $student['student_id'] ? 'selected' : '' ?>>
                                        <?= $student['section_name'] ?> - <?= $student['class_name'] ?> - 
                                        <?= $student['admission_number'] ?> - <?= $student['student_name'] ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-4 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary">Generate Statement</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <?php if ($student_data): ?>
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Student Information</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-3">
                                <p><strong>Admission No:</strong><br><?= $student_data['admission_number'] ?></p>
                            </div>
                            <div class="col-md-3">
                                <p><strong>Student Name:</strong><br><?= $student_data['student_name'] ?></p>
                            </div>
                            <div class="col-md-3">
                                <p><strong>Class:</strong><br><?= $student_data['section_name'] ?> - <?= $student_data['class_name'] ?></p>
                            </div>
                            <div class="col-md-3">
                                <p><strong>Academic Year:</strong><br><?= $student_data['academic_year'] ?></p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Fee Structure</h5>
                    </div>
                    <div class="card-body">
                        <?php if ($fee_structure): ?>
                        <div class="table-responsive">
                            <table class="table table-bordered">
                                <thead class="table-light">
                                    <tr>
                                        <th>Tuition Fee</th>
                                        <th>Development Fee</th>
                                        <th>Transportation Fee</th>
                                        <th>Other Charges</th>
                                        <th>Total Fee</th>
                                        <th>Due Date</th>
                                        <th>Installments</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td>₹<?= number_format($fee_structure['tuition_fee'], 2) ?></td>
                                        <td>₹<?= number_format($fee_structure['development_fee'], 2) ?></td>
                                        <td>₹<?= number_format($fee_structure['transportation_fee'], 2) ?></td>
                                        <td>₹<?= number_format($fee_structure['other_charges'], 2) ?></td>
                                        <td>₹<?= number_format($fee_structure['total_fee'], 2) ?></td>
                                        <td><?= date('d M Y', strtotime($fee_structure['due_date'])) ?></td>
                                        <td><?= $fee_structure['installment_allowed'] ? 'Yes ('.$fee_structure['no_of_installments'].')' : 'No' ?></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <div class="alert alert-warning">
                            No fee structure found for this student's class and academic year.
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Payment Summary</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-4">
                                <div class="card text-white bg-primary mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Fee</h6>
                                        <p class="card-text fs-4">₹<?= number_format($fee_structure ? $fee_structure['total_fee'] : 0, 2) ?></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card text-white bg-success mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Paid</h6>
                                        <p class="card-text fs-4">₹<?= number_format(array_sum(array_column($transactions, 'amount_paid')), 2) ?></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card text-white <?= $balance > 0 ? 'bg-danger' : 'bg-info' ?> mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Balance</h6>
                                        <p class="card-text fs-4">₹<?= number_format($balance, 2) ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h5>Transaction History</h5>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($transactions)): ?>
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Receipt No</th>
                                        <th>Amount</th>
                                        <th>Payment Mode</th>
                                        <th>Details</th>
                                        <th>Installment</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($transactions as $txn): ?>
                                    <tr>
                                        <td><?= date('d M Y', strtotime($txn['transaction_date'])) ?></td>
                                        <td><?= $txn['receipt_number'] ?></td>
                                        <td>₹<?= number_format($txn['amount_paid'], 2) ?></td>
                                        <td><?= $txn['payment_mode'] ?></td>
                                        <td><?= $txn['payment_details'] ?></td>
                                        <td><?= $txn['installment_number'] ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <div class="alert alert-info">
                            No transactions found for this student.
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>
            </main>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>
</body>
</html>