<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/db.php';

// Set page title
$pageTitle = "Fee Defaulters Report";

// Get academic year from query parameter or default to current
$academic_year = isset($_GET['year']) ? $_GET['year'] : date('Y') . '-' . (date('Y') + 1);

// Get all classes with fee structure
$sql_classes = "SELECT c.class_id, c.class_name, sec.section_name, fs.total_fee
                FROM classes c
                JOIN sections sec ON c.section_id = sec.section_id
                JOIN fee_structure fs ON c.class_id = fs.class_id AND fs.academic_year = ?
                ORDER BY sec.section_id, c.class_name";
$stmt_classes = $conn->prepare($sql_classes);
$stmt_classes->bind_param('s', $academic_year);
$stmt_classes->execute();
$classes_result = $stmt_classes->get_result();
$classes = $classes_result->fetch_all(MYSQLI_ASSOC);

// Get defaulters for each class
$defaulters = [];
foreach ($classes as $class) {
    // Fixed query to show all students with unpaid fees
    $sql = "SELECT s.student_id, s.admission_number, 
                   CONCAT(s.first_name, ' ', s.last_name) as student_name,
                   s.academic_year as student_academic_year,
                   IFNULL(SUM(ft.amount_paid), 0) as paid_amount,
                   ? as total_fee,
                   ? - IFNULL(SUM(ft.amount_paid), 0) as balance
            FROM students s
            LEFT JOIN fee_transactions ft ON s.student_id = ft.student_id 
                AND (ft.academic_year = ? OR ft.academic_year = s.academic_year)
            WHERE s.class_id = ? AND s.status = 'active'
            GROUP BY s.student_id, s.admission_number, s.first_name, s.last_name, s.academic_year
            HAVING balance > 0
            ORDER BY balance DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('ddsi', $class['total_fee'], $class['total_fee'], $academic_year, $class['class_id']);
    $stmt->execute();
    $result = $stmt->get_result();
    $class_defaulters = $result->fetch_all(MYSQLI_ASSOC);
    
    if (!empty($class_defaulters)) {
        $defaulters[] = [
            'class_id' => $class['class_id'],
            'class_name' => $class['class_name'],
            'section_name' => $class['section_name'],
            'total_fee' => $class['total_fee'],
            'students' => $class_defaulters
        ];
    }
}

// Get all academic years for dropdown
$sql_years = "SELECT DISTINCT academic_year FROM fee_structure ORDER BY academic_year DESC";
$years_result = $conn->query($sql_years);
$years = $years_result->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
     <?php include '../includes/header-internal.php'; ?>
</head>
<body>
 
    <div class="container-fluid">
        <div class="row">
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Fee Defaulters Report</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <div class="btn-group me-2">
                            <button class="btn btn-sm btn-outline-secondary" onclick="window.print()">
                                <i class="fas fa-print"></i> Print Report
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-6">
                                <label for="year" class="form-label">Academic Year</label>
                                <select class="form-select" id="year" name="year">
                                    <?php foreach ($years as $year): ?>
                                    <option value="<?= $year['academic_year'] ?>" <?= $year['academic_year'] == $academic_year ? 'selected' : '' ?>>
                                        <?= $year['academic_year'] ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary">Generate Report</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <?php if (empty($defaulters)): ?>
                    <div class="alert alert-info">
                        No fee defaulters found for <?= $academic_year ?> academic year.
                    </div>
                <?php else: ?>
                    <?php foreach ($defaulters as $class): ?>
                        <div class="card mb-4">
                            <div class="card-header bg-danger text-white">
                                <h5 class="mb-0">
                                    <?= $class['section_name'] ?> - <?= $class['class_name'] ?>
                                    <span class="float-end">
                                        Total Default: ₹<?= number_format(array_sum(array_column($class['students'], 'balance')), 2) ?>
                                    </span>
                                </h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Admission No</th>
                                                <th>Student Name</th>
                                                <th>Student Academic Year</th>
                                                <th>Total Fee</th>
                                                <th>Paid Amount</th>
                                                <th>Balance</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($class['students'] as $student): ?>
                                            <tr>
                                                <td><?= $student['admission_number'] ?></td>
                                                <td><?= $student['student_name'] ?></td>
                                                <td><?= $student['student_academic_year'] ?></td>
                                                <td>₹<?= number_format($class['total_fee'], 2) ?></td>
                                                <td>₹<?= number_format($student['paid_amount'], 2) ?></td>
                                                <td class="text-danger fw-bold">₹<?= number_format($student['balance'], 2) ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </main>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>
</body>
</html>