<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/db.php';

// Set page title
$pageTitle = "Collection Summary Report";

// Default date range (current month)
$start_date = date('Y-m-01');
$end_date = date('Y-m-t');
$class_id = 'all';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $start_date = $_POST['start_date'];
    $end_date = $_POST['end_date'];
    $class_id = $_POST['class_id'];
}

// Get collection summary
$sql = "SELECT DATE(ft.transaction_date) as date,
            c.class_name,
            sec.section_name,
            SUM(ft.amount_paid) as total_collected,
            COUNT(ft.transaction_id) as transaction_count
        FROM fee_transactions ft
        JOIN students s ON ft.student_id = s.student_id
        JOIN classes c ON s.class_id = c.class_id
        JOIN sections sec ON c.section_id = sec.section_id
        WHERE DATE(ft.transaction_date) BETWEEN ? AND ?";

if ($class_id != 'all') {
    $sql .= " AND s.class_id = ?";
}


          $sql .= " GROUP BY DATE(ft.transaction_date), ft.transaction_date, c.class_id, c.class_name, sec.section_name
          ORDER BY ft.transaction_date, sec.section_id, c.class_name";

$stmt = $conn->prepare($sql);

if ($class_id != 'all') {
    $stmt->bind_param('ssi', $start_date, $end_date, $class_id);
} else {
    $stmt->bind_param('ss', $start_date, $end_date);
}

$stmt->execute();
$result = $stmt->get_result();
$collections = $result->fetch_all(MYSQLI_ASSOC);

// Get total summary
$sql_total = "SELECT 
                SUM(amount_paid) as grand_total,
                COUNT(transaction_id) as total_transactions
              FROM fee_transactions
              WHERE DATE(transaction_date) BETWEEN ? AND ?";

if ($class_id != 'all') {
    $sql_total .= " AND student_id IN (SELECT student_id FROM students WHERE class_id = ?)";
}

$stmt_total = $conn->prepare($sql_total);

if ($class_id != 'all') {
    $stmt_total->bind_param('ssi', $start_date, $end_date, $class_id);
} else {
    $stmt_total->bind_param('ss', $start_date, $end_date);
}

$stmt_total->execute();
$total_result = $stmt_total->get_result();
$totals = $total_result->fetch_assoc();

// Get all classes for dropdown
$sql_classes = "SELECT c.class_id, c.class_name, sec.section_name 
                FROM classes c
                JOIN sections sec ON c.section_id = sec.section_id
                ORDER BY sec.section_id, c.class_name";
$classes_result = $conn->query($sql_classes);
$classes = $classes_result->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
   <?php include '../includes/header-internal.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
   
    <div class="container-fluid">
        <div class="row">
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Collection Summary Report</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <button class="btn btn-sm btn-outline-secondary" onclick="window.print()">
                            <i class="fas fa-print"></i> Print Report
                        </button>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <div class="col-md-3">
                                <label for="start_date" class="form-label">From Date</label>
                                <input type="date" class="form-control" id="start_date" name="start_date" 
                                       value="<?= $start_date ?>" required>
                            </div>
                            <div class="col-md-3">
                                <label for="end_date" class="form-label">To Date</label>
                                <input type="date" class="form-control" id="end_date" name="end_date" 
                                       value="<?= $end_date ?>" required>
                            </div>
                            <div class="col-md-4">
                                <label for="class_id" class="form-label">Class (Optional)</label>
                                <select class="form-select" id="class_id" name="class_id">
                                    <option value="all">All Classes</option>
                                    <?php foreach ($classes as $class): ?>
                                    <option value="<?= $class['class_id'] ?>" <?= $class_id == $class['class_id'] ? 'selected' : '' ?>>
                                        <?= $class['section_name'] ?> - <?= $class['class_name'] ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-2 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary w-100">Generate</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Summary</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-4">
                                <div class="card text-white bg-primary mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Collected</h6>
                                        <p class="card-text fs-4">₹<?= number_format($totals['grand_total'], 2) ?></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card text-white bg-success mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Transactions</h6>
                                        <p class="card-text fs-4"><?= $totals['total_transactions'] ?></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card text-white bg-info mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Date Range</h6>
                                        <p class="card-text">
                                            <?= date('d M Y', strtotime($start_date)) ?> - 
                                            <?= date('d M Y', strtotime($end_date)) ?>
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Collection Trends</h5>
                    </div>
                    <div class="card-body">
                        <canvas id="collectionChart" height="100"></canvas>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h5>Detailed Collection Report</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Class</th>
                                        <th>Section</th>
                                        <th>Amount Collected</th>
                                        <th>Transactions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($collections as $collection): ?>
                                    <tr>
                                        <td><?= date('d M Y', strtotime($collection['date'])) ?></td>
                                        <td><?= $collection['class_name'] ?></td>
                                        <td><?= $collection['section_name'] ?></td>
                                        <td>₹<?= number_format($collection['total_collected'], 2) ?></td>
                                        <td><?= $collection['transaction_count'] ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot>
                                    <tr class="table-active">
                                        <th colspan="3">Grand Total</th>
                                        <th>₹<?= number_format($totals['grand_total'], 2) ?></th>
                                        <th><?= $totals['total_transactions'] ?></th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>
    
    <script>
        // Prepare data for chart
        const dates = [...new Set(<?= json_encode(array_column($collections, 'date')) ?>)];
        const chartData = {
            labels: dates.map(date => new Date(date).toLocaleDateString()),
            datasets: [{
                label: 'Daily Collection (₹)',
                data: dates.map(date => {
                    return <?= json_encode($collections) ?>.filter(item => item.date === date)
                        .reduce((sum, item) => sum + parseFloat(item.total_collected), 0);
                }),
                backgroundColor: 'rgba(54, 162, 235, 0.2)',
                borderColor: 'rgba(54, 162, 235, 1)',
                borderWidth: 1
            }]
        };

        // Initialize chart
        document.addEventListener('DOMContentLoaded', function() {
            const ctx = document.getElementById('collectionChart').getContext('2d');
            new Chart(ctx, {
                type: 'bar',
                data: chartData,
                options: {
                    responsive: true,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Amount (₹)'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Date'
                            }
                        }
                    }
                }
            });
        });
    </script>
</body>
</html>