<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/db.php';

// Set page title
$pageTitle = "Annual Financial Report";

// Default academic year (current)
$academic_year = date('Y') . '-' . (date('Y') + 1);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $academic_year = $_POST['academic_year'];
}

// Get annual summary
$sql = "SELECT 
            MONTH(transaction_date) as month,
            SUM(amount_paid) as monthly_collection,
            COUNT(transaction_id) as transaction_count
        FROM fee_transactions
        WHERE academic_year = ?
        GROUP BY MONTH(transaction_date)
        ORDER BY MONTH(transaction_date)";

$stmt = $conn->prepare($sql);
$stmt->bind_param('s', $academic_year);
$stmt->execute();
$result = $stmt->get_result();
$monthly_data = $result->fetch_all(MYSQLI_ASSOC);

// Get total collection
$sql_total = "SELECT 
                SUM(amount_paid) as total_collection,
                COUNT(transaction_id) as total_transactions,
                COUNT(DISTINCT student_id) as total_students
              FROM fee_transactions
              WHERE academic_year = ?";

$stmt_total = $conn->prepare($sql_total);
$stmt_total->bind_param('s', $academic_year);
$stmt_total->execute();
$total_result = $stmt_total->get_result();
$totals = $total_result->fetch_assoc();

// Get class-wise collection
$sql_classes = "SELECT 
                   c.class_name,
                   sec.section_name,
                   COUNT(DISTINCT s.student_id) as total_students,
                   SUM(ft.amount_paid) as total_collected,
                   COUNT(ft.transaction_id) as transaction_count
                FROM classes c
                JOIN sections sec ON c.section_id = sec.section_id
                JOIN students s ON c.class_id = s.class_id AND s.status = 'active'
                LEFT JOIN fee_transactions ft ON s.student_id = ft.student_id AND ft.academic_year = ?
                GROUP BY c.class_id
                ORDER BY sec.section_id, c.class_name";

$stmt_classes = $conn->prepare($sql_classes);
$stmt_classes->bind_param('s', $academic_year);
$stmt_classes->execute();
$classes_result = $stmt_classes->get_result();
$class_collections = $classes_result->fetch_all(MYSQLI_ASSOC);

// Get payment mode analysis
$sql_modes = "SELECT 
                 payment_mode,
                 SUM(amount_paid) as total_amount,
                 COUNT(transaction_id) as transaction_count
              FROM fee_transactions
              WHERE academic_year = ?
              GROUP BY payment_mode
              ORDER BY total_amount DESC";

$stmt_modes = $conn->prepare($sql_modes);
$stmt_modes->bind_param('s', $academic_year);
$stmt_modes->execute();
$modes_result = $stmt_modes->get_result();
$payment_modes = $modes_result->fetch_all(MYSQLI_ASSOC);

// Get all academic years for dropdown
$sql_years = "SELECT DISTINCT academic_year FROM fee_structure ORDER BY academic_year DESC";
$years_result = $conn->query($sql_years);
$years = $years_result->fetch_all(MYSQLI_ASSOC);

// Month names for chart
$month_names = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <?php include '../includes/header-internal.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
   
    <div class="container-fluid">
        <div class="row">
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Annual Financial Report</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <button class="btn btn-sm btn-outline-secondary" onclick="window.print()">
                            <i class="fas fa-print"></i> Print Report
                        </button>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="POST" class="row g-3">
                            <div class="col-md-6">
                                <label for="academic_year" class="form-label">Academic Year</label>
                                <select class="form-select" id="academic_year" name="academic_year">
                                    <?php foreach ($years as $year): ?>
                                    <option value="<?= $year['academic_year'] ?>" <?= $year['academic_year'] == $academic_year ? 'selected' : '' ?>>
                                        <?= $year['academic_year'] ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary">Generate Report</button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Annual Summary</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-4">
                                <div class="card text-white bg-primary mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Collected</h6>
                                        <p class="card-text fs-4">₹<?= number_format($totals['total_collection'] ?? 0, 2) ?></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card text-white bg-success mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Total Transactions</h6>
                                        <p class="card-text fs-4"><?= $totals['total_transactions'] ?? 0 ?></p>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="card text-white bg-info mb-3">
                                    <div class="card-body">
                                        <h6 class="card-title">Students Paid</h6>
                                        <p class="card-text fs-4"><?= $totals['total_students'] ?? 0 ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card mb-4">
                    <div class="card-header">
                        <h5>Monthly Collection Trends</h5>
                    </div>
                    <div class="card-body">
                        <canvas id="monthlyChart" height="100"></canvas>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-6 mb-4">
                        <div class="card h-100">
                            <div class="card-header">
                                <h5>Class-wise Collection</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Class</th>
                                                <th>Students</th>
                                                <th>Amount</th>
                                                <th>% of Total</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($class_collections as $class): ?>
                                            <tr>
                                                <td><?= $class['section_name'] ?> - <?= $class['class_name'] ?></td>
                                                <td><?= $class['total_students'] ?></td>
                                                <td>₹<?= number_format($class['total_collected'] ?? 0, 2) ?></td>
                                                <td>
                                                    <?= $totals['total_collection'] > 0 ? 
                                                        number_format(($class['total_collected'] / $totals['total_collection']) * 100, 2) : '0' ?>%
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 mb-4">
                        <div class="card h-100">
                            <div class="card-header">
                                <h5>Payment Mode Analysis</h5>
                            </div>
                            <div class="card-body">
                                <canvas id="paymentModeChart" height="250"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h5>Detailed Monthly Report</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Month</th>
                                        <th>Amount Collected</th>
                                        <th>% of Annual</th>
                                        <th>Transactions</th>
                                        <th>Avg. Transaction</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    // Initialize all months with zero values
                                    $all_months = [];
                                    for ($i = 1; $i <= 12; $i++) {
                                        $all_months[$i] = [
                                            'month' => $i,
                                            'monthly_collection' => 0,
                                            'transaction_count' => 0
                                        ];
                                    }
                                    
                                    // Merge with actual data
                                    foreach ($monthly_data as $data) {
                                        $all_months[$data['month']] = $data;
                                    }
                                    
                                    // Display all months
                                    foreach ($all_months as $month): ?>
                                    <tr>
                                        <td><?= $month_names[$month['month'] - 1] ?></td>
                                        <td>₹<?= number_format($month['monthly_collection'], 2) ?></td>
                                        <td>
                                            <?= $totals['total_collection'] > 0 ? 
                                                number_format(($month['monthly_collection'] / $totals['total_collection']) * 100, 2) : '0' ?>%
                                        </td>
                                        <td><?= $month['transaction_count'] ?></td>
                                        <td>
                                            ₹<?= $month['transaction_count'] > 0 ? 
                                                number_format($month['monthly_collection'] / $month['transaction_count'], 2) : '0' ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot>
                                    <tr class="table-active">
                                        <th>Total</th>
                                        <th>₹<?= number_format($totals['total_collection'] ?? 0, 2) ?></th>
                                        <th>100%</th>
                                        <th><?= $totals['total_transactions'] ?? 0 ?></th>
                                        <th>
                                            ₹<?= $totals['total_transactions'] > 0 ? 
                                                number_format($totals['total_collection'] / $totals['total_transactions'], 2) : '0' ?>
                                        </th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <?php include '../includes/footer.php'; ?>
    
    <script>
        // Prepare data for charts
        const monthlyLabels = <?= json_encode($month_names) ?>;
        const monthlyData = <?= json_encode(array_column($all_months, 'monthly_collection')) ?>;
        
        const paymentModeLabels = <?= json_encode(array_column($payment_modes, 'payment_mode')) ?>;
        const paymentModeData = <?= json_encode(array_column($payment_modes, 'total_amount')) ?>;
        const paymentModeColors = ['#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF', '#FF9F40'];

        // Initialize charts
        document.addEventListener('DOMContentLoaded', function() {
            // Monthly Collection Chart
            const monthlyCtx = document.getElementById('monthlyChart').getContext('2d');
            new Chart(monthlyCtx, {
                type: 'line',
                data: {
                    labels: monthlyLabels,
                    datasets: [{
                        label: 'Monthly Collection (₹)',
                        data: monthlyData,
                        backgroundColor: 'rgba(54, 162, 235, 0.2)',
                        borderColor: 'rgba(54, 162, 235, 1)',
                        borderWidth: 2,
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    scales: {
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: 'Amount (₹)'
                            }
                        },
                        x: {
                            title: {
                                display: true,
                                text: 'Month'
                            }
                        }
                    }
                }
            });
            
            // Payment Mode Chart
            const paymentModeCtx = document.getElementById('paymentModeChart').getContext('2d');
            new Chart(paymentModeCtx, {
                type: 'doughnut',
                data: {
                    labels: paymentModeLabels,
                    datasets: [{
                        data: paymentModeData,
                        backgroundColor: paymentModeColors
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: {
                            position: 'bottom',
                        },
                        title: {
                            display: true,
                            text: 'By Payment Mode (₹)'
                        }
                    }
                }
            });
        });
    </script>
</body>
</html>