// main.js - School Fee Management System

// 1. Database Models (simplified)
class Student {
  constructor(id, name, grade, contactInfo) {
    this.id = id;
    this.name = name;
    this.grade = grade;
    this.contactInfo = contactInfo;
    this.feeRecords = [];
  }
}

class FeeRecord {
  constructor(studentId, amount, dueDate, paymentStatus = 'unpaid') {
    this.studentId = studentId;
    this.amount = amount;
    this.dueDate = dueDate;
    this.paymentStatus = paymentStatus;
    this.paymentDate = null;
    this.transactionId = null;
  }
}

// 2. Main Application Class
class FeeManagementSystem {
  constructor() {
    this.students = [];
    this.feeRecords = [];
    this.feeStructure = {
      'Grade 1': 5000,
      'Grade 2': 5500,
      // ... other grades
    };
  }

  // 3. Core Methods
  addStudent(studentData) {
    const newStudent = new Student(
      this.generateId(),
      studentData.name,
      studentData.grade,
      studentData.contactInfo
    );
    this.students.push(newStudent);
    this.generateFeeRecords(newStudent.id, studentData.grade);
    return newStudent;
  }

  generateFeeRecords(studentId, grade) {
    const termFees = this.feeStructure[grade];
    const terms = ['Term 1', 'Term 2', 'Term 3'];
    
    terms.forEach(term => {
      const dueDate = this.calculateDueDate(term);
      const feeRecord = new FeeRecord(studentId, termFees, dueDate);
      this.feeRecords.push(feeRecord);
      
      // Add reference to student
      const student = this.students.find(s => s.id === studentId);
      if (student) {
        student.feeRecords.push(feeRecord);
      }
    });
  }

  recordPayment(studentId, term, paymentData) {
    const record = this.feeRecords.find(fr => 
      fr.studentId === studentId && fr.term === term
    );
    
    if (record) {
      record.paymentStatus = 'paid';
      record.paymentDate = new Date();
      record.transactionId = paymentData.transactionId;
      return true;
    }
    return false;
  }

  generateReport(type, filters = {}) {
    // Could be 'outstanding', 'paid', 'all' etc.
    switch(type) {
      case 'outstanding':
        return this.feeRecords.filter(fr => fr.paymentStatus === 'unpaid');
      case 'paid':
        return this.feeRecords.filter(fr => fr.paymentStatus === 'paid');
      default:
        return this.feeRecords;
    }
  }

  // Helper methods
  generateId() {
    return 'STD-' + Math.random().toString(36).substr(2, 9);
  }

  calculateDueDate(term) {
    // Simplified - would use actual dates in real implementation
    const now = new Date();
    switch(term) {
      case 'Term 1': return new Date(now.getFullYear(), 2, 1); // March 1
      case 'Term 2': return new Date(now.getFullYear(), 6, 1); // July 1
      case 'Term 3': return new Date(now.getFullYear(), 10, 1); // November 1
    }
  }
}

// 4. Initialize the system
const feeSystem = new FeeManagementSystem();

// Example usage
const newStudent = feeSystem.addStudent({
  name: 'John Doe',
  grade: 'Grade 1',
  contactInfo: {
    parentName: 'Jane Doe',
    phone: '123-456-7890',
    email: 'parent@example.com'
  }
});

// Record payment
feeSystem.recordPayment(newStudent.id, 'Term 1', {
  transactionId: 'PAY-12345',
  amount: 5000,
  method: 'bank transfer'
});

// Generate report
const outstandingFees = feeSystem.generateReport('outstanding');